<?php
/**
 * Settings Page Class.
 *
 * Handles plugin settings and admin interface.
 *
 * @package WP_WebP_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings Class.
 */
class WP_WebP_Optimizer_Settings {

	/**
	 * Option name for plugin settings.
	 *
	 * @var string
	 */
	private $option_name = 'wp_webp_optimizer_settings';

	/**
	 * Initialize settings.
	 *
	 * @since 1.0.0
	 */
	public function init() {
		// Add admin menu
		add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
		
		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		
		// Enqueue admin assets
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		
		// Add settings link on plugins page
		add_filter( 'plugin_action_links_' . SMALL_IMAGE_PLUGIN_BASENAME, array( $this, 'add_action_links' ) );
	}

	/**
	 * Add settings page to admin menu.
	 *
	 * @since 1.0.0
	 */
	public function add_settings_page() {
		// Add main menu page (Dashboard)
		add_menu_page(
			__( 'Small Image - Dashboard', 'small-image' ),
			__( 'Small Image', 'Mak8it.com' ),
			'manage_options',
			'small-image',
			array( $this, 'render_settings_page' ),
			$this->get_menu_icon(),
			65
		);

		// Add Dashboard submenu (rename the first item)
		add_submenu_page(
			'small-image',
			__( 'Dashboard', 'Mak8it.com' ),
			__( 'Dashboard', 'Mak8it.com' ),
			'manage_options',
			'small-image',
			array( $this, 'render_settings_page' )
		);

		// Add Bulk Converter submenu
		add_submenu_page(
			'small-image',
			__( 'Bulk Converter', 'Mak8it.com' ),
			__( 'Bulk Converter', 'Mak8it.com' ),
			'manage_options',
			'small-image-bulk',
			array( $this, 'render_bulk_page' )
		);

		// Add System Info submenu
		add_submenu_page(
			'small-image',
			__( 'System Info', 'Mak8it.com' ),
			__( 'System Info', 'Mak8it.com' ),
			'manage_options',
			'small-image-system-info',
			array( $this, 'render_system_info_page' )
		);
	}

	/**
	 * Get menu icon.
	 *
	 * @since 1.0.0
	 * @return string Dashicon class name.
	 */
	private function get_menu_icon() {
		return 'dashicons-format-image';
	}

	/**
	 * Register plugin settings.
	 *
	 * @since 1.0.0
	 */
	public function register_settings() {
		// Register setting
		register_setting(
			'wp_webp_optimizer_settings_group',
			$this->option_name,
			array( $this, 'sanitize_settings' )
		);

		// Add settings section
	add_settings_section(
		'wp_webp_optimizer_main_section',
		'', // No title
		array( $this, 'render_section_description' ),
		'small-image'
	);

		// Auto-convert field
		add_settings_field(
			'auto_convert',
			__( 'Auto-Convert on Upload', 'Mak8it.com' ),
			array( $this, 'render_auto_convert_field' ),
			'small-image',
			'wp_webp_optimizer_main_section'
		);

		// Quality field
		add_settings_field(
			'quality',
			__( 'Conversion Quality', 'Mak8it.com' ),
			array( $this, 'render_quality_field' ),
			'small-image',
			'wp_webp_optimizer_main_section'
		);

		// Replace original field
		add_settings_field(
			'replace_original',
			__( 'File Handling', 'Mak8it.com' ),
			array( $this, 'render_replace_field' ),
			'small-image',
			'wp_webp_optimizer_main_section'
		);

		// Show WebP in library field
		add_settings_field(
			'show_webp_in_library',
			__( 'Media Library Display', 'Mak8it.com' ),
			array( $this, 'render_show_webp_field' ),
			'small-image',
			'wp_webp_optimizer_main_section'
		);

		// Frontend serving field
		add_settings_field(
			'enable_frontend_serving',
			__( 'Frontend Serving', 'Mak8it.com' ),
			array( $this, 'render_frontend_serving_field' ),
			'small-image',
			'wp_webp_optimizer_main_section'
		);
	}

	/**
	 * Render section description.
	 *
	 * @since 1.0.0
	 */
	public function render_section_description() {
		// Check WebP support
		$support = WP_WebP_Optimizer_Converter::check_webp_support();
		
		echo '<div class="webp-optimizer-support-status">';
		if ( $support['supported'] ) {
			echo '<p class="webp-support-yes">';
			echo '<span class="dashicons dashicons-yes-alt"></span> ';
			echo esc_html( $support['message'] );
			echo ' <strong>(' . esc_html( $support['method'] ) . ')</strong>';
			echo '</p>';
		} else {
			echo '<p class="webp-support-no">';
			echo '<span class="dashicons dashicons-warning"></span> ';
			echo esc_html( $support['message'] );
			echo '</p>';
		}
		echo '</div>';
	}

	/**
	 * Render auto-convert field.
	 *
	 * @since 1.0.0
	 */
	public function render_auto_convert_field() {
		$settings = get_option( $this->option_name, array() );
		$value    = isset( $settings['auto_convert'] ) ? $settings['auto_convert'] : '1';
		?>
		<label>
			<input type="checkbox" 
				   name="<?php echo esc_attr( $this->option_name ); ?>[auto_convert]" 
				   value="1" 
				   <?php checked( $value, '1' ); ?>>
			<?php esc_html_e( 'Automatically convert images to WebP when uploaded', 'Mak8it.com' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'Enable this to automatically convert JPEG and PNG images to WebP format during upload.', 'Mak8it.com' ); ?>
		</p>
		<?php
	}

	/**
	 * Render quality field.
	 *
	 * @since 1.0.0
	 */
	public function render_quality_field() {
		$settings = get_option( $this->option_name, array() );
		$value    = isset( $settings['quality'] ) ? intval( $settings['quality'] ) : 60;
		?>
		<div class="quality-slider-container">
			<input type="range" 
				   id="quality-slider"
				   name="<?php echo esc_attr( $this->option_name ); ?>[quality]" 
				   min="10" 
				   max="100" 
				   value="<?php echo esc_attr( $value ); ?>"
				   step="5">
			<span class="quality-value"><?php echo esc_html( $value ); ?>%</span>
		</div>
		<p class="description">
			<?php esc_html_e( 'Set the quality for WebP conversion. Lower values = smaller file size but lower quality. Recommended: 60-80%', 'Mak8it.com' ); ?>
		</p>
		<?php
	}

	/**
	 * Render replace original field.
	 *
	 * @since 1.0.0
	 */
	public function render_replace_field() {
		$settings = get_option( $this->option_name, array() );
		$value    = isset( $settings['replace_original'] ) ? $settings['replace_original'] : 'keep_both';
		?>
		<fieldset>
			<label>
				<input type="radio" 
					   name="<?php echo esc_attr( $this->option_name ); ?>[replace_original]" 
					   value="keep_both" 
					   <?php checked( $value, 'keep_both' ); ?>>
				<?php esc_html_e( 'Keep both original and WebP files', 'Mak8it.com' ); ?>
			</label>
			<br>
			<label>
				<input type="radio" 
					   name="<?php echo esc_attr( $this->option_name ); ?>[replace_original]" 
					   value="replace" 
					   <?php checked( $value, 'replace' ); ?>>
				<?php esc_html_e( 'Replace original with WebP file', 'Mak8it.com' ); ?>
			</label>
		</fieldset>
		<p class="description">
			<?php esc_html_e( 'Choose whether to keep the original files or replace them with WebP versions.', 'Mak8it.com' ); ?>
		</p>
		<?php
	}

	/**
	 * Render show WebP in library field.
	 *
	 * @since 1.0.0
	 */
	public function render_show_webp_field() {
		$settings = get_option( $this->option_name, array() );
		$value    = isset( $settings['show_webp_in_library'] ) ? $settings['show_webp_in_library'] : '1';
		?>
		<label>
			<input type="checkbox" 
				   name="<?php echo esc_attr( $this->option_name ); ?>[show_webp_in_library]" 
				   value="1" 
				   <?php checked( $value, '1' ); ?>>
			<?php esc_html_e( 'Show WebP files as separate items in Media Library', 'Mak8it.com' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'When "Keep both files" is selected, this will create separate media library entries for WebP versions. Disable this if you only want the original files to appear in the library (WebP files will still be created in the background).', 'Mak8it.com' ); ?>
		</p>
		<?php
	}

	/**
	 * Render frontend serving field.
	 *
	 * @since 1.0.0
	 */
	public function render_frontend_serving_field() {
		$settings = get_option( $this->option_name, array() );
		$value    = isset( $settings['enable_frontend_serving'] ) ? $settings['enable_frontend_serving'] : '1';
		?>
		<label>
			<input type="checkbox" 
				   name="<?php echo esc_attr( $this->option_name ); ?>[enable_frontend_serving]" 
				   value="1" 
				   <?php checked( $value, '1' ); ?>>
			<?php esc_html_e( 'Serve WebP images on frontend', 'Mak8it.com' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'Automatically replace image URLs with WebP versions on the frontend if they exist. This uses output buffering to rewrite HTML.', 'Mak8it.com' ); ?>
		</p>
		<?php
	}

	/**
	 * Sanitize settings before saving.
	 *
	 * @since 1.0.0
	 * @param array $input Raw input data.
	 * @return array Sanitized data.
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// Sanitize auto_convert (checkbox)
		$sanitized['auto_convert'] = isset( $input['auto_convert'] ) ? '1' : '0';

		// Sanitize quality (must be 10-100)
		if ( isset( $input['quality'] ) ) {
			$quality = intval( $input['quality'] );
			$sanitized['quality'] = max( 10, min( 100, $quality ) );
		} else {
			$sanitized['quality'] = 60;
		}

		// Sanitize replace_original (must be one of two values)
		if ( isset( $input['replace_original'] ) && in_array( $input['replace_original'], array( 'keep_both', 'replace' ), true ) ) {
			$sanitized['replace_original'] = sanitize_text_field( $input['replace_original'] );
		} else {
			$sanitized['replace_original'] = 'keep_both';
		}

		// Sanitize show_webp_in_library (checkbox)
		$sanitized['show_webp_in_library'] = isset( $input['show_webp_in_library'] ) ? '1' : '0';

		// Sanitize enable_frontend_serving (checkbox)
		$sanitized['enable_frontend_serving'] = isset( $input['enable_frontend_serving'] ) ? '1' : '0';

		return $sanitized;
	}

	/**
	 * Render settings page.
	 *
	 * @since 1.0.0
	 */
	public function render_settings_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'Mak8it.com' ) );
		}

		// Load template
		require_once SMALL_IMAGE_PLUGIN_DIR . 'admin/settings-page.php';
	}


	/**
	 * Render bulk converter page.
	 *
	 * @since 1.0.0
	 */
	public function render_bulk_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'Mak8it.com' ) );
		}

		// Load template
		require_once SMALL_IMAGE_PLUGIN_DIR . 'admin/bulk-converter.php';
	}

	/**
	 * Render system info page.
	 *
	 * @since 1.0.0
	 */
	public function render_system_info_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'Mak8it.com' ) );
		}

		// Load template
		require_once SMALL_IMAGE_PLUGIN_DIR . 'admin/system-info.php';
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @since 1.0.0
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		// Load on settings page
		if ( 'toplevel_page_small-image' === $hook ) {
			wp_enqueue_style(
				'small-image-admin',
				SMALL_IMAGE_PLUGIN_URL . 'admin/css/admin-style.css',
				array(),
				SMALL_IMAGE_VERSION
			);

			wp_enqueue_script(
				'small-image-admin',
				SMALL_IMAGE_PLUGIN_URL . 'admin/js/admin-script.js',
				array( 'jquery' ),
				SMALL_IMAGE_VERSION,
				true
			);
		}

		// Load bulk converter styles on all plugin pages
		$plugin_pages = array(
			'toplevel_page_small-image',
			'small-image_page_small-image-bulk',
			'small-image_page_small-image-system-info',
		);

		if ( in_array( $hook, $plugin_pages, true ) ) {
			wp_enqueue_style(
				'small-image-bulk',
				SMALL_IMAGE_PLUGIN_URL . 'admin/css/bulk-style.css',
				array(),
				SMALL_IMAGE_VERSION
			);
		}

		// Load bulk converter page JavaScript
		if ( 'small-image_page_small-image-bulk' === $hook ) {
			wp_enqueue_script(
				'small-image-bulk',
				SMALL_IMAGE_PLUGIN_URL . 'admin/js/bulk-converter.js',
				array( 'jquery' ),
				SMALL_IMAGE_VERSION,
				true
			);

			// Localize script with AJAX data
			wp_localize_script(
				'small-image-bulk',
				'webpBulkData',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( 'webp_bulk_nonce' ),
					'strings' => array(
						'startingConversion' => __( 'Starting conversion...', 'Mak8it.com' ),
						'processing'         => __( 'Processing...', 'Mak8it.com' ),
						'completed'          => __( 'Conversion completed!', 'Mak8it.com' ),
						'paused'             => __( 'Conversion paused', 'Mak8it.com' ),
						'stopped'            => __( 'Conversion stopped', 'Mak8it.com' ),
						'error'              => __( 'An error occurred', 'Mak8it.com' ),
						'confirmStop'        => __( 'Are you sure you want to stop the conversion? Progress will be lost.', 'Mak8it.com' ),
						'noFilters'          => __( 'No filters applied', 'Mak8it.com' ),
					),
				)
			);
		}
	}

	/**
	 * Add action links on plugins page.
	 *
	 * @since 1.0.0
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_action_links( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'admin.php?page=small-image' ),
			__( 'Dashboard', 'Mak8it.com' )
		);

		$bulk_link = sprintf(
			'<a href="%s" style="color: #00a32a; font-weight: 600;">%s</a>',
			admin_url( 'admin.php?page=small-image-bulk' ),
			__( 'Bulk Convert', 'Mak8it.com' )
		);
		
		array_unshift( $links, $settings_link, $bulk_link );
		
		return $links;
	}
}

